﻿//=======================================================================================
// Waves.cpp Frank Luna (C) 2008 Wszelkie prawa zastrzeżone.
//=======================================================================================

#include "Waves.h"
#include <algorithm>
#include <vector>
#include <cassert>

Waves::Waves()
: mNumRows(0), mNumCols(0), mVertexCount(0), mTriangleCount(0), 
  mK1(0.0f), mK2(0.0f), mK3(0.0f), mTimeStep(0.0f), mSpatialStep(0.0f),
  mPrevSolution(0), mCurrSolution(0)
{
}

Waves::~Waves()
{
	delete[] mPrevSolution;
	delete[] mCurrSolution;
}

UINT Waves::RowCount()const
{
	return mNumRows;
}

UINT Waves::ColumnCount()const
{
	return mNumCols;
}

UINT Waves::VertexCount()const
{
	return mVertexCount;
}

UINT Waves::TriangleCount()const
{
	return mTriangleCount;
}

void Waves::Init(UINT m, UINT n, float dx, float dt, float speed, float damping)
{
	mNumRows  = m;
	mNumCols  = n;

	mVertexCount   = m*n;
	mTriangleCount = (m-1)*(n-1)*2;

	mTimeStep    = dt;
	mSpatialStep = dx;

	float d = damping*dt+2.0f;
	float e = (speed*speed)*(dt*dt)/(dx*dx);
	mK1     = (damping*dt-2.0f)/ d;
	mK2     = (4.0f-8.0f*e) / d;
	mK3     = (2.0f*e) / d;

	// W razie ponownego wywołania Init().
	delete[] mPrevSolution;
	delete[] mCurrSolution;

	mPrevSolution = new XMFLOAT3[m*n];
	mCurrSolution = new XMFLOAT3[m*n];

	// Generuj wierzchołki siatki w pamięci systemowej.

	float halfWidth = (n-1)*dx*0.5f;
	float halfDepth = (m-1)*dx*0.5f;
	for(UINT i = 0; i < m; ++i)
	{
		float z = halfDepth - i*dx;
		for(UINT j = 0; j < n; ++j)
		{
			float x = -halfWidth + j*dx;

			mPrevSolution[i*n+j] = XMFLOAT3(x, 0.0f, z);
			mCurrSolution[i*n+j] = XMFLOAT3(x, 0.0f, z);
		}
	}
}

void Waves::Update(float dt)
{
	static float t = 0;

	// Oblicz skumulowany czas.
	t += dt;

	// Aktualizuj symulację dopiero po przekroczeniu określonego progu czasu.
	if( t >= mTimeStep )
	{
		// Aktualizuj tylko punkty wewnętrzne.
		for(DWORD i = 1; i < mNumRows-1; ++i)
		{
			for(DWORD j = 1; j < mNumCols-1; ++j)
			{
				// Po tej aktualizacji nie będziemy już potrzebować wcześniejszego
				// poprzedniego bufora, zapisujemy w nim zatem nowe dane.
				// Zauważ, że wykorzystujemy tu jednoczesny odczyt i zapis do
				// tego samego elementu. Jest to możliwe, ponieważ przypisanie do zmiennej
				// następuje na końcu. 

				// Zauważ, że j indeksuje x a i indeksuje z: h(x_j, z_i, t_k)
				// Ponadto oś +z skierowana jest w dół, dzięki czemu
				// zachowana jest zgodność z indeksami wierszowymi.

				mPrevSolution[i*mNumCols+j].y = 
					mK1*mPrevSolution[i*mNumCols+j].y +
					mK2*mCurrSolution[i*mNumCols+j].y +
					mK3*(mCurrSolution[(i+1)*mNumCols+j].y + 
					     mCurrSolution[(i-1)*mNumCols+j].y + 
					     mCurrSolution[i*mNumCols+j+1].y + 
						 mCurrSolution[i*mNumCols+j-1].y);
			}
		}

		// W buforze zostały zapisane nowe dane. Stają się one bieżącym rozwiązaniem
		// (mCurrSolution) a wcześniejsze bieżące rozwiązanie staje się rozwiązaniem 
		// poprzednim (mPrevSolution).
		std::swap(mPrevSolution, mCurrSolution);

		t = 0.0f; // resetuj czas
	}
}

void Waves::Disturb(UINT i, UINT j, float magnitude)
{
	// Nie wzburzaj granic.
	assert(i > 1 && i < mNumRows-2);
	assert(j > 1 && j < mNumCols-2);

	float halfMag = 0.5f*magnitude;

	// Wzburz wierzchołek ij i jego sąsiadów (nadając odpowiednią wysokość).
	mCurrSolution[i*mNumCols+j].y     += magnitude;
	mCurrSolution[i*mNumCols+j+1].y   += halfMag;
	mCurrSolution[i*mNumCols+j-1].y   += halfMag;
	mCurrSolution[(i+1)*mNumCols+j].y += halfMag;
	mCurrSolution[(i-1)*mNumCols+j].y += halfMag;
}
	
